document.addEventListener("DOMContentLoaded", async () => {
  const container = document.getElementById("table-list");

  const tabs = await browser.tabs.query({ active: true, currentWindow: true });
  const tab = tabs && tabs[0];

  if (!tab || !tab.id || !tab.url) {
    container.innerText = "No active tab found.";
    return;
  }

  const isToolsClub = tab.url.includes("toolsclub.net") || tab.url.includes("semrush");

  // ❗️БОЛЬШЕ НЕ ДЕЛАЕМ executeScript — content.js уже подключён через manifest
  let response;
  try {
    response = await browser.tabs.sendMessage(tab.id, { action: "getTables" });
  } catch (e) {
    container.innerText =
      "Can't read tables on this page (blocked page, or content script didn't run).";
    return;
  }

  if (!response || !response.tables || !response.tables.length) {
    container.innerText = "No tables found on this page.";
    return;
  }

  container.innerHTML = "";

  response.tables.forEach((table, tIndex) => {
    const div = document.createElement("div");
    div.className = "table-container";

    const tableTitle = table.name ? table.name : `Table ${tIndex + 1}`;
    div.innerHTML = `<h2>${escapeHtml(tableTitle)}</h2>`;

    const rows = table.rows || [];
    const colCount = Math.max(...rows.map((r) => r.length), 0);

    // Preview table with copy icons
    const tableEl = document.createElement("table");
    tableEl.style.borderCollapse = "collapse";
    tableEl.style.width = "100%";

    const thead = tableEl.createTHead();
    const headRow = thead.insertRow();

    for (let c = 0; c < colCount; c++) {
      const th = document.createElement("th");

      const img = document.createElement("img");
      img.src = "copy.jpg";
      img.style.width = "18px";
      img.style.cursor = "pointer";
      img.title = `Copy column ${c + 1}`;

      img.onclick = async () => {
        const colData = rows.map((r) => r[c] || "").join("\n");
        try {
          await navigator.clipboard.writeText(colData);
        } catch (e) {
          alert("Clipboard write failed. Try granting clipboard permissions.");
        }
      };

      th.appendChild(img);
      headRow.appendChild(th);
    }

    const tbody = tableEl.createTBody();
    rows.forEach((r) => {
      const tr = tbody.insertRow();
      for (let c = 0; c < colCount; c++) {
        const td = tr.insertCell();
        td.innerText = r[c] || "";
        td.style.border = "1px solid #ccc";
        td.style.padding = "4px";
      }
    });

    div.appendChild(tableEl);

    // Export CSV
    const btnCsv = document.createElement("button");
    btnCsv.innerText = "export CSV";
    btnCsv.style.marginTop = "10px";
    btnCsv.style.padding = "6px 10px";
    btnCsv.style.cursor = "pointer";

    btnCsv.addEventListener("click", async () => {
      const filename = isToolsClub ? "SEMrush_table.csv" : `table_${tIndex + 1}.csv`;
      const ok = await exportTableToCSV(rows, filename, isToolsClub);
      if (!ok) alert("Download failed (CSV). Check about:debugging console.");
    });

    div.appendChild(btnCsv);

    // Export REAL XLS (BIFF8)
    const btnXls = document.createElement("button");
    btnXls.innerText = "export XLS";
    btnXls.style.marginTop = "10px";
    btnXls.style.marginLeft = "10px";
    btnXls.style.padding = "6px 10px";
    btnXls.style.cursor = "pointer";
    btnXls.style.backgroundColor = "#2e7d32";
    btnXls.style.color = "white";
    btnXls.style.border = "none";
    btnXls.style.borderRadius = "6px";

    btnXls.addEventListener("click", async () => {
      const filename = isToolsClub ? "SEMrush_table.xls" : `table_${tIndex + 1}.xls`;
      const ok = await exportTableToRealXLS(rows, filename, isToolsClub, tableTitle);
      if (!ok) alert("Download failed (XLS). Check about:debugging console.");
    });

    div.appendChild(btnXls);

    container.appendChild(div);
  });
});

// =========================
// CSV export (semicolon ;)
// =========================
async function exportTableToCSV(rows, filename, isToolsClub) {
  try {
    if (!filename.toLowerCase().endsWith(".csv")) filename += ".csv";

    const allRows = buildAllRows(rows, isToolsClub);

    const csvBody = allRows
      .map((row) =>
        row
          .map((v) => {
            const val = v == null ? "" : v.toString();
            return /[;"\n\r]/.test(val) ? `"${val.replace(/"/g, '""')}"` : val;
          })
          .join(";")
      )
      .join("\r\n");

    const csvContent = "\uFEFF" + "sep=;\r\n" + csvBody;

    const data = new TextEncoder().encode(csvContent).buffer;

    const res = await browser.runtime.sendMessage({
      action: "downloadFile",
      filename,
      mime: "text/csv;charset=utf-8",
      data,
      saveAs: true
    });

    if (!res || !res.ok) {
      console.error("CSV download failed:", res);
      return false;
    }
    return true;
  } catch (e) {
    console.error("CSV export error:", e);
    return false;
  }
}

// ==========================================
// REAL XLS (BIFF8) export using SheetJS
// ==========================================
async function exportTableToRealXLS(rows, filename, isToolsClub, sheetName = "Table") {
  try {
    if (!filename.toLowerCase().endsWith(".xls")) filename += ".xls";

    if (!window.XLSX) {
      alert("SheetJS (XLSX) library not loaded. Include xlsx.full.min.js in popup.html before popup.js");
      return false;
    }

    const allRows = buildAllRows(rows, isToolsClub);

    const normalized = allRows.map((row, rIdx) =>
      row.map((cell) => {
        if (rIdx === 0) return (cell ?? "").toString();
        const s = (cell ?? "").toString().trim();
        const raw = s.replace(/,/g, "");
        const num = Number(raw);
        return s !== "" && Number.isFinite(num) && /^-?\d+(\.\d+)?$/.test(raw)
          ? num
          : (cell ?? "").toString();
      })
    );

    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.aoa_to_sheet(normalized);

    const safeSheet = (sheetName || "Table").toString().slice(0, 31);
    XLSX.utils.book_append_sheet(wb, ws, safeSheet);

    // IMPORTANT: write to ArrayBuffer instead of XLSX.writeFile
    const arrayBuffer = XLSX.write(wb, { bookType: "biff8", type: "array" });

    const res = await browser.runtime.sendMessage({
      action: "downloadFile",
      filename,
      mime: "application/vnd.ms-excel",
      data: arrayBuffer,
      saveAs: true
    });

    if (!res || !res.ok) {
      console.error("XLS download failed:", res);
      return false;
    }
    return true;
  } catch (e) {
    console.error("XLS export error:", e);
    return false;
  }
}

// ========= helpers =========
function buildAllRows(rows, isToolsClub) {
  if (isToolsClub) {
    const headers = ["Domain", "Authority", "Visits"];
    const data = rows.map((row) => {
      const target = (row[0] || "").split("\n")[0].trim();
      let authority = (row[1] || "0").toString();
      let visits = (row[4] || "0").toString();
      if (authority.toLowerCase() === "n/a") authority = "0";
      if (visits.toLowerCase() === "n/a") visits = "0";
      return [target, authority, visits];
    });
    return [headers, ...data];
  }

  const maxCols = Math.max(...rows.map((r) => r.length), 0);
  const headers = Array.from({ length: maxCols }, (_, i) => `Column ${i + 1}`);
  return [headers, ...rows.map((r) => r.map((v) => (v ?? "").toString()))];
}

function escapeHtml(str) {
  return (str ?? "")
    .toString()
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#039;");
}
